// Admin Panel JavaScript

class AdminPanel {
    constructor() {
        this.initAdminPanel();
        this.loadSettings();
        this.bindEvents();
    }
    
    initAdminPanel() {
        // Create admin panel toggle
        const adminToggle = document.createElement('button');
        adminToggle.className = 'admin-toggle';
        adminToggle.innerHTML = '<i class="fas fa-cogs"></i> ADMIN';
        adminToggle.addEventListener('click', () => this.toggleAdminPanel());
        
        const adminPanel = document.createElement('div');
        adminPanel.id = 'admin-panel';
        adminPanel.className = 'admin-panel hidden';
        adminPanel.innerHTML = `
            <div class="admin-header">
                <h3>Control Panel</h3>
                <button class="close-admin">&times;</button>
            </div>
            <div class="admin-content">
                <div class="admin-section">
                    <h4>Section Order</h4>
                    <div class="section-list" id="sectionList"></div>
                    <button id="saveOrder">Save Order</button>
                </div>
                <div class="admin-section">
                    <h4>Theme Settings</h4>
                    <label>Primary Color: <input type="color" id="primaryColor"></label>
                    <label>Secondary Color: <input type="color" id="secondaryColor"></label>
                    <button id="saveTheme">Save Theme</button>
                </div>
                <div class="admin-section">
                    <h4>Quick Actions</h4>
                    <button id="addFood">Add Food Item</button>
                    <button id="uploadImage">Upload Image</button>
                    <button id="viewOrders">View Orders</button>
                </div>
            </div>
        `;
        
        document.body.appendChild(adminToggle);
        document.body.appendChild(adminPanel);
        
        this.panel = adminPanel;
        this.toggle = adminToggle;
    }
    
    toggleAdminPanel() {
        this.panel.classList.toggle('hidden');
    }
    
    async loadSettings() {
        try {
            const response = await fetch('/api/settings');
            const settings = await response.json();
            this.applySettings(settings);
        } catch (error) {
            console.error('Error loading settings:', error);
        }
    }
    
    applySettings(settings) {
        // Apply theme colors
        if (settings.primaryColor) {
            document.documentElement.style.setProperty('--neon-blue', settings.primaryColor);
        }
        if (settings.secondaryColor) {
            document.documentElement.style.setProperty('--neon-pink', settings.secondaryColor);
        }
        
        // Apply section order
        if (settings.sectionOrder) {
            this.reorderSections(settings.sectionOrder);
        }
    }
    
    reorderSections(order) {
        const container = document.querySelector('main');
        const sections = Array.from(container.children);
        
        order.forEach(sectionId => {
            const section = sections.find(s => s.id === sectionId);
            if (section) {
                container.appendChild(section);
            }
        });
    }
    
    bindEvents() {
        // Save section order
        document.getElementById('saveOrder').addEventListener('click', () => this.saveSectionOrder());
        
        // Save theme
        document.getElementById('saveTheme').addEventListener('click', () => this.saveTheme());
        
        // Add food item
        document.getElementById('addFood').addEventListener('click', () => this.showAddFoodForm());
        
        // Close admin panel
        this.panel.querySelector('.close-admin').addEventListener('click', () => {
            this.panel.classList.add('hidden');
        });
    }
    
    async saveSectionOrder() {
        const sections = Array.from(document.querySelectorAll('section'));
        const order = sections.map(section => section.id);
        
        try {
            await fetch('/api/settings/section-order', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({ order })
            });
            
            this.showNotification('Section order saved!');
        } catch (error) {
            this.showNotification('Error saving order', 'error');
        }
    }
    
    async saveTheme() {
        const primaryColor = document.getElementById('primaryColor').value;
        const secondaryColor = document.getElementById('secondaryColor').value;
        
        try {
            await fetch('/api/settings/theme', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({ primaryColor, secondaryColor })
            });
            
            this.showNotification('Theme saved!');
        } catch (error) {
            this.showNotification('Error saving theme', 'error');
        }
    }
    
    showAddFoodForm() {
        const formHTML = `
            <div class="modal-overlay">
                <div class="modal">
                    <h3>Add Food Item</h3>
                    <form id="addFoodForm">
                        <input type="text" placeholder="Food Name" required>
                        <textarea placeholder="Description" required></textarea>
                        <input type="number" placeholder="Price" step="0.01" required>
                        <input type="number" placeholder="Discount Price" step="0.01">
                        <input type="number" placeholder="Preparation Time (minutes)" required>
                        <input type="file" accept="image/*" id="foodImage">
                        <button type="submit">Add Food</button>
                    </form>
                </div>
            </div>
        `;
        
        document.body.insertAdjacentHTML('beforeend', formHTML);
        
        document.getElementById('addFoodForm').addEventListener('submit', (e) => {
            e.preventDefault();
            this.saveFoodItem();
        });
    }
    
    async saveFoodItem() {
        const form = document.getElementById('addFoodForm');
        const formData = new FormData(form);
        
        try {
            const response = await fetch('/api/foods', {
                method: 'POST',
                body: formData
            });
            
            if (response.ok) {
                this.showNotification('Food item added!');
                document.querySelector('.modal-overlay').remove();
            }
        } catch (error) {
            this.showNotification('Error adding food', 'error');
        }
    }
    
    showNotification(message, type = 'success') {
        const notification = document.createElement('div');
        notification.className = `admin-notification ${type}`;
        notification.textContent = message;
        
        this.panel.appendChild(notification);
        
        setTimeout(() => {
            notification.remove();
        }, 3000);
    }
}

// Initialize admin panel when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    // Check if user is admin
    const isAdmin = localStorage.getItem('isAdmin') === 'true';
    
    if (isAdmin) {
        new AdminPanel();
    }
});

// Admin login function
async function adminLogin(email, password) {
    try {
        const response = await fetch('/api/admin/login', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ email, password })
        });
        
        if (response.ok) {
            const data = await response.json();
            localStorage.setItem('adminToken', data.token);
            localStorage.setItem('isAdmin', 'true');
            location.reload();
        }
    } catch (error) {
        console.error('Login error:', error);
    }
}

// Admin logout function
function adminLogout() {
    localStorage.removeItem('adminToken');
    localStorage.removeItem('isAdmin');
    location.reload();
}