// Main frontend functionality

document.addEventListener('DOMContentLoaded', function() {
    // Initialize components
    initFoodMenu();
    initGallery();
    initContactForm();
    initOrderTimer();
});

// Food Menu Management
function initFoodMenu() {
    const foodGrid = document.querySelector('.food-grid');
    
    // Sample food data (will be replaced with API call)
    const foods = [
        {
            id: 1,
            name: "Neural Noodles",
            description: "Synth-enhanced ramen with glowing broth",
            price: 24.99,
            discountPrice: 19.99,
            prepTime: 15,
            image: "https://images.unsplash.com/photo-1557872943-16a5ac26437e?w=400&h=300&fit=crop"
        },
        {
            id: 2,
            name: "Cyborg Burger",
            description: "Lab-grown patty with digital sauce",
            price: 29.99,
            discountPrice: null,
            prepTime: 20,
            image: "https://images.unsplash.com/photo-1568901346375-23c9450c58cd?w=400&h=300&fit=crop"
        }
    ];

    foods.forEach(food => {
        const card = createFoodCard(food);
        foodGrid.appendChild(card);
    });
}

function createFoodCard(food) {
    const card = document.createElement('div');
    card.className = 'food-card';
    card.innerHTML = `
        <img src="${food.image}" alt="${food.name}" class="food-image">
        <h3 class="food-name">${food.name}</h3>
        <p class="food-description">${food.description}</p>
        <div class="food-price">
            ${food.discountPrice ? 
                `<span class="original-price">$${food.price}</span>
                 <span class="discounted-price">$${food.discountPrice}</span>` :
                `<span class="current-price">$${food.price}</span>`
            }
        </div>
        <p class="prep-time"><i class="fas fa-clock"></i> ${food.prepTime} mins</p>
        <button class="add-to-cart" data-id="${food.id}">
            <i class="fas fa-shopping-cart"></i> ADD TO CART
        </button>
    `;
    
    card.querySelector('.add-to-cart').addEventListener('click', () => addToCart(food));
    
    card.addEventListener('click', (e) => {
        if (!e.target.classList.contains('add-to-cart')) {
            showFoodDetails(food);
        }
    });
    
    return card;
}

function addToCart(food) {
    const cart = JSON.parse(localStorage.getItem('cart') || '[]');
    cart.push({
        id: food.id,
        name: food.name,
        price: food.discountPrice || food.price,
        prepTime: food.prepTime
    });
    localStorage.setItem('cart', JSON.stringify(cart));
    
    showNotification(`${food.name} added to cart!`);
}

function showFoodDetails(food) {
    const modal = document.createElement('div');
    modal.className = 'food-modal';
    modal.innerHTML = `
        <div class="modal-content">
            <span class="close-modal">&times;</span>
            <img src="${food.image}" alt="${food.name}">
            <h3>${food.name}</h3>
            <p>${food.description}</p>
            <p>Preparation Time: ${food.prepTime} minutes</p>
            <button class="btn btn-primary add-to-cart-modal" data-id="${food.id}">
                Add to Cart - $${food.discountPrice || food.price}
            </button>
        </div>
    `;
    
    document.body.appendChild(modal);
    
    modal.querySelector('.close-modal').addEventListener('click', () => modal.remove());
    modal.querySelector('.add-to-cart-modal').addEventListener('click', () => {
        addToCart(food);
        modal.remove();
    });
}

// Gallery Management
function initGallery() {
    const galleryGrid = document.querySelector('.gallery-grid');
    
    // Sample images (will be replaced with API call)
    const images = [
        "https://images.unsplash.com/photo-1517248135467-4c7edcad34c4?w=400&h-300&fit=crop",
        "https://images.unsplash.com/photo-1555396273-367ea4eb4db5?w=400&h=300&fit=crop",
        "https://images.unsplash.com/photo-1424847651672-bf20a4b0982b?w=400&h=300&fit=crop"
    ];
    
    images.forEach(img => {
        const item = document.createElement('div');
        item.className = 'gallery-item';
        item.innerHTML = `<img src="${img}" alt="Restaurant Image">`;
        galleryGrid.appendChild(item);
    });
}

// Contact Form
function initContactForm() {
    const contactForm = document.getElementById('contactForm');
    if (contactForm) {
        contactForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const formData = new FormData(contactForm);
            const data = Object.fromEntries(formData);
            
            try {
                const response = await fetch('/api/contact', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify(data)
                });
                
                if (response.ok) {
                    showNotification('Message sent successfully!');
                    contactForm.reset();
                }
            } catch (error) {
                showNotification('Error sending message', 'error');
            }
        });
    }
}

// Order Timer
function initOrderTimer() {
    const orderTimer = localStorage.getItem('currentOrderTimer');
    if (orderTimer) {
        const orderData = JSON.parse(orderTimer);
        startOrderTimer(orderData.endTime);
    }
}

function startOrderTimer(endTime) {
    const timerInterval = setInterval(() => {
        const now = new Date().getTime();
        const distance = endTime - now;
        
        if (distance < 0) {
            clearInterval(timerInterval);
            showNotification('Your order is ready!');
            localStorage.removeItem('currentOrderTimer');
            return;
        }
        
        const minutes = Math.floor((distance % (1000 * 60 * 60)) / (1000 * 60));
        const seconds = Math.floor((distance % (1000 * 60)) / 1000);
        
        updateTimerDisplay(minutes, seconds);
    }, 1000);
}

function updateTimerDisplay(minutes, seconds) {
    let timerElement = document.getElementById('order-timer');
    if (!timerElement) {
        timerElement = document.createElement('div');
        timerElement.id = 'order-timer';
        timerElement.className = 'order-timer';
        document.body.appendChild(timerElement);
    }
    
    timerElement.innerHTML = `
        <div class="timer-container">
            <h3>Current Order</h3>
            <div class="timer">${minutes}:${seconds.toString().padStart(2, '0')}</div>
            <p>Estimated delivery time</p>
        </div>
    `;
}

// Notification System
function showNotification(message, type = 'success') {
    const notification = document.createElement('div');
    notification.className = `notification ${type}`;
    notification.textContent = message;
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.classList.add('show');
    }, 10);
    
    setTimeout(() => {
        notification.classList.remove('show');
        setTimeout(() => {
            notification.remove();
        }, 300);
    }, 3000);
}

// Order System
function placeOrder(orderData) {
    return fetch('/api/orders', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify(orderData)
    });
}

// User Authentication
async function registerUser(email, password) {
    const response = await fetch('/api/register', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({ email, password })
    });
    
    return response.json();
}

async function loginUser(email, password) {
    const response = await fetch('/api/login', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({ email, password })
    });
    
    return response.json();
}

// Export functions for use in other modules
window.restaurantApp = {
    addToCart,
    placeOrder,
    registerUser,
    loginUser,
    showNotification
};