const express = require('express');
const router = express.Router();
const multer = require('multer');
const Food = require('../models/Food');
const Order = require('../models/Order');
const Settings = require('../models/Settings');

// Configure multer for file uploads
const storage = multer.diskStorage({
    destination: (req, file, cb) => {
        cb(null, 'uploads/');
    },
    filename: (req, file, cb) => {
        cb(null, Date.now() + '-' + file.originalname);
    }
});

const upload = multer({ storage });

// Admin middleware
const isAdmin = (req, res, next) => {
    if (req.session.isAdmin) {
        next();
    } else {
        res.redirect('/admin/login');
    }
};

// Admin dashboard
router.get('/dashboard', isAdmin, async (req, res) => {
    try {
        const [totalOrders, pendingOrders, foods, recentOrders] = await Promise.all([
            Order.countDocuments(),
            Order.countDocuments({ status: 'pending' }),
            Food.countDocuments(),
            Order.find().sort({ createdAt: -1 }).limit(5)
        ]);
        
        res.render('admin/dashboard', {
            totalOrders,
            pendingOrders,
            foods,
            recentOrders
        });
    } catch (error) {
        res.status(500).send('Server error');
    }
});

// Food management
router.get('/foods', isAdmin, async (req, res) => {
    try {
        const foods = await Food.find().sort({ createdAt: -1 });
        res.render('admin/menu', { foods });
    } catch (error) {
        res.status(500).send('Server error');
    }
});

router.post('/foods', isAdmin, upload.single('image'), async (req, res) => {
    try {
        const food = new Food({
            ...req.body,
            image: req.file ? '/uploads/' + req.file.filename : null
        });
        await food.save();
        res.redirect('/admin/foods');
    } catch (error) {
        res.status(500).send('Error adding food');
    }
});

// Order management
router.get('/orders', isAdmin, async (req, res) => {
    try {
        const orders = await Order.find()
            .sort({ createdAt: -1 })
            .populate('userId', 'email name');
        res.render('admin/orders', { orders });
    } catch (error) {
        res.status(500).send('Server error');
    }
});

// Gallery management
router.get('/gallery', isAdmin, (req, res) => {
    res.render('admin/gallery');
});

router.post('/gallery', isAdmin, upload.array('images'), async (req, res) => {
    try {
        const settings = await Settings.findOne();
        const galleryImages = settings.galleryImages || [];
        
        req.files.forEach(file => {
            galleryImages.push('/uploads/' + file.filename);
        });
        
        settings.galleryImages = galleryImages;
        await settings.save();
        
        res.redirect('/admin/gallery');
    } catch (error) {
        res.status(500).send('Error uploading images');
    }
});

// Settings management
router.get('/settings', isAdmin, async (req, res) => {
    try {
        const settings = await Settings.findOne();
        res.render('admin/settings', { settings });
    } catch (error) {
        res.status(500).send('Server error');
    }
});

router.post('/settings', isAdmin, async (req, res) => {
    try {
        let settings = await Settings.findOne();
        if (!settings) {
            settings = new Settings(req.body);
        } else {
            Object.assign(settings, req.body);
        }
        await settings.save();
        res.redirect('/admin/settings');
    } catch (error) {
        res.status(500).send('Error saving settings');
    }
});

module.exports = router;