const express = require('express');
const router = express.Router();
const Food = require('../models/Food');
const Order = require('../models/Order');
const User = require('../models/User');
const Settings = require('../models/Settings');
const multer = require('multer');
const path = require('path');

// Configure storage for file uploads
const storage = multer.diskStorage({
    destination: function (req, file, cb) {
        cb(null, 'uploads/')
    },
    filename: function (req, file, cb) {
        cb(null, Date.now() + path.extname(file.originalname))
    }
});

const upload = multer({ storage: storage });

// Get all food items
router.get('/foods', async (req, res) => {
    try {
        const foods = await Food.find({ isAvailable: true });
        res.json(foods);
    } catch (error) {
        res.status(500).json({ error: 'Server error' });
    }
});

// Get single food item
router.get('/foods/:id', async (req, res) => {
    try {
        const food = await Food.findById(req.params.id);
        if (!food) {
            return res.status(404).json({ error: 'Food not found' });
        }
        res.json(food);
    } catch (error) {
        res.status(500).json({ error: 'Server error' });
    }
});

// Create new food item
router.post('/foods', upload.single('image'), async (req, res) => {
    try {
        const foodData = req.body;
        if (req.file) {
            foodData.image = '/uploads/' + req.file.filename;
        }
        
        const food = new Food(foodData);
        await food.save();
        res.status(201).json(food);
    } catch (error) {
        res.status(500).json({ error: 'Error creating food item' });
    }
});

// Update food item
router.put('/foods/:id', upload.single('image'), async (req, res) => {
    try {
        const foodData = req.body;
        if (req.file) {
            foodData.image = '/uploads/' + req.file.filename;
        }
        
        const food = await Food.findByIdAndUpdate(
            req.params.id,
            foodData,
            { new: true }
        );
        
        res.json(food);
    } catch (error) {
        res.status(500).json({ error: 'Error updating food item' });
    }
});

// Delete food item
router.delete('/foods/:id', async (req, res) => {
    try {
        await Food.findByIdAndDelete(req.params.id);
        res.json({ message: 'Food item deleted' });
    } catch (error) {
        res.status(500).json({ error: 'Error deleting food item' });
    }
});

// Place order
router.post('/orders', async (req, res) => {
    try {
        const { items, customerInfo, paymentMethod } = req.body;
        
        // Calculate total amount
        let totalAmount = 0;
        let maxPrepTime = 0;
        
        for (const item of items) {
            const food = await Food.findById(item.foodId);
            if (food) {
                totalAmount += item.quantity * (food.discountPrice || food.price);
                maxPrepTime = Math.max(maxPrepTime, food.prepTime);
            }
        }
        
        // Add delivery charge (get from settings)
        const settings = await Settings.findOne();
        const deliveryCharge = settings.deliveryCharges || 5;
        totalAmount += deliveryCharge;
        
        // Add tax
        const taxRate = settings.taxRate || 0.1;
        totalAmount += totalAmount * taxRate;
        
        // Create order
        const order = new Order({
            items,
            totalAmount,
            deliveryCharge,
            estimatedTime: maxPrepTime + 30, // Add 30 minutes for delivery
            customerInfo,
            paymentMethod
        });
        
        // Create user account if not exists
        let user = await User.findOne({ email: customerInfo.email });
        if (!user) {
            user = new User({
                email: customerInfo.email,
                name: customerInfo.name,
                phone: customerInfo.phone,
                password: Math.random().toString(36).slice(-8) // Generate random password
            });
            await user.save();
        }
        
        order.userId = user._id;
        await order.save();
        
        // Send order confirmation email (simulated)
        // In production, integrate with email service
        
        res.status(201).json({
            order,
            user: {
                id: user._id,
                email: user.email,
                password: user.password // Only for demo, in production send via email
            }
        });
    } catch (error) {
        console.error('Error placing order:', error);
        res.status(500).json({ error: 'Error placing order' });
    }
});

// Get user orders
router.get('/users/:userId/orders', async (req, res) => {
    try {
        const orders = await Order.find({ userId: req.params.userId })
            .sort({ createdAt: -1 });
        res.json(orders);
    } catch (error) {
        res.status(500).json({ error: 'Error fetching orders' });
    }
});

// Update order status
router.put('/orders/:id/status', async (req, res) => {
    try {
        const { status } = req.body;
        const order = await Order.findByIdAndUpdate(
            req.params.id,
            { status },
            { new: true }
        );
        
        res.json(order);
    } catch (error) {
        res.status(500).json({ error: 'Error updating order status' });
    }
});

// Submit review
router.post('/foods/:id/reviews', async (req, res) => {
    try {
        const { userId, rating, comment, images } = req.body;
        
        // Check if user purchased this food
        const hasPurchased = await Order.exists({
            userId,
            'items.foodId': req.params.id,
            status: 'delivered'
        });
        
        if (!hasPurchased) {
            return res.status(403).json({ error: 'You must purchase this item to review it' });
        }
        
        const food = await Food.findById(req.params.id);
        
        // Update rating
        const newTotalRatings = food.totalRatings + 1;
        const newRating = ((food.rating * food.totalRatings) + rating) / newTotalRatings;
        
        food.rating = newRating;
        food.totalRatings = newTotalRatings;
        
        // Add review
        if (!food.reviews) food.reviews = [];
        food.reviews.push({
            userId,
            rating,
            comment,
            images,
            date: new Date()
        });
        
        await food.save();
        res.json(food);
    } catch (error) {
        res.status(500).json({ error: 'Error submitting review' });
    }
});

// Get settings
router.get('/settings', async (req, res) => {
    try {
        let settings = await Settings.findOne();
        if (!settings) {
            settings = new Settings();
            await settings.save();
        }
        res.json(settings);
    } catch (error) {
        res.status(500).json({ error: 'Error fetching settings' });
    }
});

// Update settings
router.put('/settings', async (req, res) => {
    try {
        let settings = await Settings.findOne();
        if (!settings) {
            settings = new Settings(req.body);
        } else {
            settings = Object.assign(settings, req.body);
        }
        await settings.save();
        res.json(settings);
    } catch (error) {
        res.status(500).json({ error: 'Error updating settings' });
    }
});

// Upload gallery images
router.post('/gallery', upload.array('images', 10), async (req, res) => {
    try {
        const settings = await Settings.findOne();
        const images = req.files.map(file => '/uploads/' + file.filename);
        
        if (!settings.gallery) settings.gallery = [];
        settings.gallery.push(...images);
        
        await settings.save();
        res.json({ images });
    } catch (error) {
        res.status(500).json({ error: 'Error uploading images' });
    }
});

// Contact form submission
router.post('/contact', async (req, res) => {
    try {
        const { name, email, subject, message } = req.body;
        
        // Save to database or send email
        // For now, just return success
        res.json({ 
            success: true, 
            message: 'Message received. We will contact you soon!' 
        });
    } catch (error) {
        res.status(500).json({ error: 'Error sending message' });
    }
});

module.exports = router;