const mongoose = require('mongoose');
const bcrypt = require('bcryptjs');
require('dotenv').config();

// Models
const User = require('./models/User');
const Food = require('./models/Food');
const Settings = require('./models/Settings');

const seedDatabase = async () => {
    try {
        await mongoose.connect(process.env.MONGODB_URI);
        console.log('Connected to MongoDB');
        
        // Clear existing data
        await User.deleteMany({});
        await Food.deleteMany({});
        await Settings.deleteMany({});
        
        // Create admin user
        const salt = await bcrypt.genSalt(10);
        const hashedPassword = await bcrypt.hash(process.env.ADMIN_PASSWORD, salt);
        
        const adminUser = new User({
            email: process.env.ADMIN_EMAIL,
            password: hashedPassword,
            name: 'Admin',
            isAdmin: true
        });
        await adminUser.save();
        console.log('Admin user created');
        
        // Create sample foods
        const sampleFoods = [
            {
                name: 'Neural Noodles',
                description: 'Synth-enhanced ramen with glowing broth and digital vegetables',
                price: 24.99,
                discountPrice: 19.99,
                prepTime: 15,
                category: 'main',
                ingredients: ['Synth-Noodles', 'Glowing Broth', 'Digital Veggies', 'Cyber-Spices'],
                image: 'https://images.unsplash.com/photo-1557872943-16a5ac26437e?w=400&h=300&fit=crop'
            },
            {
                name: 'Cyborg Burger',
                description: 'Lab-grown patty with digital sauce and cyber-bun',
                price: 29.99,
                prepTime: 20,
                category: 'main',
                ingredients: ['Lab-Grown Patty', 'Digital Sauce', 'Cyber-Bun', 'Synth-Cheese'],
                image: 'https://images.unsplash.com/photo-1568901346375-23c9450c58cd?w=400&h=300&fit=crop'
            },
            {
                name: 'Quantum Pizza',
                description: 'Multi-dimensional pizza with quantum toppings',
                price: 32.99,
                discountPrice: 28.99,
                prepTime: 25,
                category: 'main',
                ingredients: ['Multi-Dough', 'Quantum Cheese', 'Digital Pepperoni', 'Holographic Veggies'],
                image: 'https://images.unsplash.com/photo-1565299624946-b28f40a0ae38?w=400&h=300&fit=crop'
            },
            {
                name: 'Neon Smoothie',
                description: 'Glowing smoothie with energy-boosting cyber-fruits',
                price: 12.99,
                prepTime: 5,
                category: 'drink',
                ingredients: ['Neon Berries', 'Energy Fruits', 'Cyber-Milk', 'Glowing Ice'],
                image: 'https://images.unsplash.com/photo-1505252585461-04db1eb84625?w=400&h=300&fit=crop'
            },
            {
                name: 'Hologram Cake',
                description: '3D holographic dessert that changes flavor',
                price: 18.99,
                prepTime: 30,
                category: 'dessert',
                ingredients: ['Hologram Batter', 'Flavor Crystals', 'Light Frosting', 'Digital Sprinkles'],
                image: 'https://images.unsplash.com/photo-1563729784474-d77dbb933a9e?w=400&h=300&fit=crop'
            }
        ];
        
        await Food.insertMany(sampleFoods);
        console.log('Sample foods created');
        
        // Create default settings
        const defaultSettings = new Settings({
            restaurantName: 'NEOBITE',
            primaryColor: '#00f3ff',
            secondaryColor: '#ff00ff',
            theme: 'cyberpunk',
            email: 'info@neobite.com',
            phone: '+1 (555) CYBER-DINE',
            address: 'Neo District, Cyber City 2077',
            hours: 'Open 24/7',
            deliveryCharges: 5,
            taxRate: 0.1,
            header: {
                menuItems: [
                    { name: 'Home', link: '#home', order: 1 },
                    { name: 'Menu', link: '#menu', order: 2 },
                    { name: 'Order', link: '#order', order: 3 },
                    { name: 'Gallery', link: '#gallery', order: 4 },
                    { name: 'Reviews', link: '#reviews', order: 5 },
                    { name: 'Contact', link: '#contact', order: 6 }
                ],
                ctaButtons: [
                    { text: 'Login', link: '/auth/login', type: 'secondary' },
                    { text: 'Sign Up', link: '/auth/register', type: 'primary' }
                ]
            },
            heroSection: {
                title: 'NEOBITE',
                subtitle: 'WHERE CYBERPUNK MEETS CULINARY EXCELLENCE',
                videoUrl: 'https://assets.mixkit.co/videos/preview/mixkit-abstract-digital-grid-5385-large.mp4',
                buttonText: 'ORDER NOW'
            },
            sectionOrder: ['hero', 'menu', 'gallery', 'reviews', 'contact'],
            socialLinks: [
                { platform: 'facebook', url: 'https://facebook.com', icon: 'fab fa-facebook-f' },
                { platform: 'twitter', url: 'https://twitter.com', icon: 'fab fa-twitter' },
                { platform: 'instagram', url: 'https://instagram.com', icon: 'fab fa-instagram' },
                { platform: 'discord', url: 'https://discord.com', icon: 'fab fa-discord' }
            ]
        });
        
        await defaultSettings.save();
        console.log('Default settings created');
        
        console.log('\n✅ Database seeded successfully!');
        console.log('\n📋 Access Information:');
        console.log(`   Admin Email: ${process.env.ADMIN_EMAIL}`);
        console.log(`   Admin Password: ${process.env.ADMIN_PASSWORD}`);
        console.log(`   Frontend: http://localhost:${process.env.PORT || 3000}`);
        console.log(`   Admin Panel: http://localhost:${process.env.PORT || 3000}/admin/dashboard`);
        
        process.exit(0);
    } catch (error) {
        console.error('Error seeding database:', error);
        process.exit(1);
    }
};

seedDatabase();