const express = require('express');
const mongoose = require('mongoose');
const session = require('express-session');
const MongoStore = require('connect-mongo');
const path = require('path');
const fs = require('fs');
require('dotenv').config();

const app = express();

// Create uploads directory if it doesn't exist
const uploadsDir = path.join(__dirname, 'uploads');
if (!fs.existsSync(uploadsDir)) {
    fs.mkdirSync(uploadsDir, { recursive: true });
}

// Middleware
app.use(express.json());
app.use(express.urlencoded({ extended: true }));
app.use(express.static('public'));
app.use('/uploads', express.static('uploads'));

// Session configuration
app.use(session({
    secret: process.env.SESSION_SECRET || 'cyberpunk-secret',
    resave: false,
    saveUninitialized: false,
    store: MongoStore.create({
        mongoUrl: process.env.MONGODB_URI || 'mongodb://localhost:27017/cyberpunk_restaurant'
    }),
    cookie: { 
        maxAge: 1000 * 60 * 60 * 24,
        secure: process.env.NODE_ENV === 'production',
        httpOnly: true
    }
}));

// View engine
app.set('view engine', 'ejs');
app.set('views', [
    path.join(__dirname, 'views'),
    path.join(__dirname, 'views/admin'),
    path.join(__dirname, 'views/user')
]);

// Database connection
mongoose.connect(process.env.MONGODB_URI || 'mongodb://localhost:27017/cyberpunk_restaurant', {
    useNewUrlParser: true,
    useUnifiedTopology: true
});

const db = mongoose.connection;
db.on('error', console.error.bind(console, 'MongoDB connection error:'));
db.once('open', () => {
    console.log('Connected to MongoDB');
    
    // Create default admin if not exists
    createDefaultAdmin();
});

// Import models
const Settings = require('./models/Settings');

// Middleware to pass settings to all views
app.use(async (req, res, next) => {
    try {
        let settings = await Settings.findOne();
        if (!settings) {
            settings = new Settings();
            await settings.save();
        }
        res.locals.settings = settings;
    } catch (error) {
        console.error('Error loading settings:', error);
    }
    next();
});

// Import routes
const adminRoutes = require('./routes/admin');
const apiRoutes = require('./routes/api');
const authRoutes = require('./routes/auth');
const uploadRoutes = require('./routes/uploads');

// Use routes
app.use('/admin', adminRoutes);
app.use('/api', apiRoutes);
app.use('/auth', authRoutes);
app.use('/api/upload', uploadRoutes);

// User routes
app.get('/user/dashboard', async (req, res) => {
    if (!req.session.userId) {
        return res.redirect('/auth/login');
    }
    
    try {
        const User = require('./models/User');
        const user = await User.findById(req.session.userId);
        
        if (!user) {
            req.session.destroy();
            return res.redirect('/auth/login');
        }
        
        res.render('user/dashboard', { user });
    } catch (error) {
        console.error('Error loading user dashboard:', error);
        res.redirect('/auth/login');
    }
});

// Main route
app.get('/', async (req, res) => {
    try {
        const Food = require('./models/Food');
        const foods = await Food.find({ isAvailable: true }).limit(6);
        
        res.render('index', { 
            foods,
            user: req.session.userId ? { id: req.session.userId } : null
        });
    } catch (error) {
        console.error('Error loading home page:', error);
        res.render('index', { foods: [], user: null });
    }
});

// Auth pages
app.get('/auth/login', (req, res) => {
    res.render('auth/login');
});

app.get('/auth/register', (req, res) => {
    res.render('auth/register');
});

// Error handling middleware
app.use((err, req, res, next) => {
    console.error(err.stack);
    res.status(500).render('error', { error: err.message });
});

// 404 handler
app.use((req, res) => {
    res.status(404).render('404');
});

const PORT = process.env.PORT || 3000;
app.listen(PORT, () => {
    console.log(`Server running on port ${PORT}`);
    console.log(`Admin panel: http://localhost:${PORT}/admin/dashboard`);
    console.log(`Default admin: ${process.env.ADMIN_EMAIL} / ${process.env.ADMIN_PASSWORD}`);
});

// Helper function to create default admin
async function createDefaultAdmin() {
    try {
        const User = require('./models/User');
        const bcrypt = require('bcryptjs');
        
        const adminEmail = process.env.ADMIN_EMAIL || 'admin@neobite.com';
        const adminPassword = process.env.ADMIN_PASSWORD || 'admin123';
        
        const existingAdmin = await User.findOne({ email: adminEmail });
        
        if (!existingAdmin) {
            const salt = await bcrypt.genSalt(10);
            const hashedPassword = await bcrypt.hash(adminPassword, salt);
            
            const adminUser = new User({
                email: adminEmail,
                password: hashedPassword,
                name: 'Admin',
                isAdmin: true
            });
            
            await adminUser.save();
            console.log('Default admin user created');
        }
    } catch (error) {
        console.error('Error creating default admin:', error);
    }
}